﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Data.SqlClient;
using System.Data;
using System.Net;
using System.IO;

namespace AHOBPR_Job_Runner
{
    public class DataMigration : LoggedJob
    {
        protected override String LogFileName { get { return "AHOBPR_Job_Runner.log"; } }
        public override string Id { get { return "DM"; } }

        public override void Run()
        {
            JsonFormParser formParser = new JsonFormParser();
            JsonParserJob jsonParser = new JsonParserJob();

            base.StartLog();
            LogMessageToFile("Starting AHOBPR Data Migration...");
            base.CheckDiskSpace();

            //Set Mode to Migration
            UpdateServiceMode(true);

            //Bring over the form json:
            MoveFormJsonFromFileToJson();
            PopulateJsonQueue();
            formParser.Run();

            //Run Post Migration update scripts:
            runPostMigrationScript();

            //Reset form lookup global variables
            ResetFormLookupGlobalVariables();

            //Save the latest form ID
            setFormId();

            //Bring user & info JSON objects over:
            UpdateJsonQueueForResponses(false);
            jsonParser.Run();

            //Bring responses objects over + anything was missed:
            UpdateJsonQueueForResponses(true);
            jsonParser.Run();

            //Process the additional info objects updates for contact preferences
            jsonParser.Run();

            //Set Mode to Normal
            UpdateServiceMode(false);

            LogMessageToFile("AHOBPR Data Migration complete.");
        }


        private void UpdateJsonQueueForResponses(bool include)
        {
            //   string sqlConnectionString = System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"];
            string sqlConnectionString = System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"];
            using (SqlConnection dbConnection = new SqlConnection(sqlConnectionString))
            {
                try
                {
                    dbConnection.Open();
                    if (dbConnection.State == ConnectionState.Open)
                    {
                        LogMessageToFile("The SqlConnection is open.");
                    }
                    else
                    {
                        LogMessageToFile("ERROR: Unabale to open a connection to SQL.");
                        return;
                    }

                    LogMessageToFile("");

                    string includeMessage = include ? "include" : "exclude";
                    LogMessageToFile("Update Json Queue to " + includeMessage + " responses");

                    String sqlText = string.Empty;
                    if (include)
                    {
                        sqlText = "update [AHOBPR].JSON_QUEUE set PROCESS_START_DATE = null, PROCESS_STATUS = null ";
                        sqlText = sqlText + "WHERE [REGISTRANT_JSON_ID] IN ( SELECT [REGISTRANT_JSON_ID] FROM [AHOBPR].[REGISTRANT_JSON] where OBJECT_TYPE = 'responses') ";
                    }
                    else
                    {
                        sqlText = "update [AHOBPR].JSON_QUEUE set PROCESS_START_DATE = '1/1/1900', PROCESS_STATUS = 'success' ";
                        sqlText = sqlText + "WHERE [REGISTRANT_JSON_ID] IN ( SELECT [REGISTRANT_JSON_ID] FROM [AHOBPR].[REGISTRANT_JSON] where OBJECT_TYPE = 'responses') ";
                    }
                    SqlCommand sqlCmd = new SqlCommand(sqlText, dbConnection);

                    sqlCmd = new SqlCommand(sqlText, dbConnection);
                    sqlCmd.ExecuteNonQuery();

                    dbConnection.Close();
                    LogMessageToFile("The SqlConnection is closed.");
                    LogMessageToFile("UpdateJsonQueueForResponses is completed.");
                }
                catch (Exception ex)
                {
                    LogErrorMessage(ex);
                    LogMessageToFile("UpdateJsonQueueForResponses is unsuccessful!");
                    return;
                }
            }
        }

        private void MoveFormJsonFromFileToJson()
        {
         //   string sqlConnectionString = System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"];
            string sqlConnectionString = System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"];
            using (SqlConnection dbConnection = new SqlConnection(sqlConnectionString))
            {
                try
                {
                    dbConnection.Open();
                    if (dbConnection.State == ConnectionState.Open)
                    {
                        LogMessageToFile("The SqlConnection is open.");
                    }
                    else
                    {
                        LogMessageToFile("ERROR: Unabale to open a connection to SQL.");
                        return;
                    }

                    LogMessageToFile("");

                    LogMessageToFile("Moving Form JSON from the REGISTRANT_FILE table to the REGISTRANT_JSON table...");

                    int totalRecords = 0;
                    String sqlText = "select file_id, file_content from AHOBPR.REGISTRANT_FILE where file_type = 'form' ";
                    SqlCommand sqlCmd = new SqlCommand(sqlText, dbConnection);
                    Dictionary<String, String> allFormJson = new Dictionary<String, String>();
                    try
                    {
                        //Get all the file content:
                        using (SqlDataReader dataReader = sqlCmd.ExecuteReader())
                        {
                            String formJson;
                            String formId;
                            while (dataReader.Read())
                            {
                                formId = dataReader[0].ToString();
                                formJson = System.Text.Encoding.UTF8.GetString((byte[])dataReader[1]).Replace("\\\"", "\"");
                                formJson = formJson.Substring(12, formJson.Length - 14);
                                formJson = formJson.Replace("’", "'");
                                formJson = formJson.Replace("“", "\\\"");
                                formJson = formJson.Replace("”", "\\\"");
                                formJson = formJson.Replace("…", "...");
                                allFormJson.Add(formId, formJson);
                                totalRecords++;
                            }
                        }

                        //Insert all the entries into REGISTRANT_JSON:
                        sqlText = "insert into AHOBPR.REGISTRANT_JSON (object_id, object_type, json_data, createdby, updatedby, created, updated) \r\n " +
                            " values (@objectId, @objectType, @jsonData, @created, @created, getDate(), getDate()) ";
                        foreach (String formId in allFormJson.Keys)
                        {
                            sqlCmd = new SqlCommand(sqlText, dbConnection);
                            sqlCmd.Parameters.Add(new SqlParameter() { ParameterName = "@objectId", Value =  formId});
                            sqlCmd.Parameters.Add(new SqlParameter() { ParameterName = "@objectType", Value = "form" });
                            sqlCmd.Parameters.Add(new SqlParameter() { ParameterName = "@jsonData", Value = allFormJson[formId] });
                            sqlCmd.Parameters.Add(new SqlParameter() { ParameterName = "@created", Value = "RESTful Service" });
                            sqlCmd.ExecuteNonQuery();
                        }

                        //Delete all form entries from the REGISTRANT_FILE table:
                        sqlText = "delete from AHOBPR.REGISTRANT_FILE where file_type = 'form' ";
                        sqlCmd = new SqlCommand(sqlText, dbConnection);
                        sqlCmd.ExecuteNonQuery();

                        dbConnection.Close();
                        LogMessageToFile("Processed " + totalRecords + " form records.");
                        LogMessageToFile("The SqlConnection is closed.");
                        LogMessageToFile("AHOBPR Form JSON move is completed.");
                    }
                    catch (Exception e)
                    {
                        LogErrorMessage(e);
                    }
                }
                catch (Exception ex)
                {
                    LogErrorMessage(ex);
                    LogMessageToFile("AHOBPR Data Migration execution unsuccessful!");
                    return;
                }
            }
        }

        private void setFormId()
        {
            String latestFormId = null;
            string sqlText = "select top 1 BPR_FORM_ID \r\n " + 
                            "from ahobpr.STD_FORM \r\t" + 
                            "order by VERSION desc";

           // string sqlConnectionString = System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"];
            string sqlConnectionString = System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"];
            using (SqlConnection dbConnection = new SqlConnection(sqlConnectionString))
            {
                dbConnection.Open();
                SqlCommand sqlCmd = new SqlCommand(sqlText, dbConnection);
                try
                {
                    latestFormId = sqlCmd.ExecuteScalar().ToString();
                    LogMessageToFile("Got the latest BPR Form ID.");
                }
                catch (Exception e)
                {
                    LogErrorMessage(e);
                }
            }

            if (!String.IsNullOrEmpty(latestFormId)){
                WebRequest request = WebRequest.Create(System.Configuration.ConfigurationManager.AppSettings["RestServiceUrl"].Replace("jsonparser/", "") + "updatedefaultformid/" + latestFormId);
                request.Method = "GET";
                request.ContentType = @"application/json; charset=utf-8";
                String secretKey = System.Configuration.ConfigurationManager.AppSettings["secretKey"];
                request.Headers.Add("Authorization", "Basic " + secretKey);

                HttpWebResponse response = (HttpWebResponse)request.GetResponse();
                string jsonResponse = string.Empty;
                using (StreamReader sr = new StreamReader(response.GetResponseStream()))
                {
                    jsonResponse = sr.ReadToEnd();
                    LogMessageToFile("Response from REST service for default form id update: " + jsonResponse);
                }
            }            
        }

        private void ResetFormLookupGlobalVariables()
        {
            WebRequest request = WebRequest.Create(System.Configuration.ConfigurationManager.AppSettings["RestServiceUrl"] + "resetglobal");
            LogMessageToFile("REST service url: " + System.Configuration.ConfigurationManager.AppSettings["RestServiceUrl"] + "resetglobal");
            request.Method = "GET";
            request.ContentType = @"application/json; charset=utf-8";

            String secretKey = System.Configuration.ConfigurationManager.AppSettings["secretKey"];
            request.Headers.Add("Authorization", "Basic " + secretKey);

            HttpWebResponse response = (HttpWebResponse)request.GetResponse();
            string jsonResponse = string.Empty;
            using (StreamReader sr = new StreamReader(response.GetResponseStream()))
            {
                jsonResponse = sr.ReadToEnd();
                LogMessageToFile("Response from REST service: " + jsonResponse);
            }

        }

        private void PopulateJsonQueue()
        {
            //    string sqlConnectionString = System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"];
            string sqlConnectionString = System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"];
            using (SqlConnection dbConnection = new SqlConnection(sqlConnectionString))
            {
                dbConnection.Open();
                SqlCommand sqlCmd = new SqlCommand("AHOBPR.SP_LoadJsonQueue", dbConnection);
                try
                {
                    sqlCmd.CommandType = CommandType.StoredProcedure;
                    sqlCmd.ExecuteNonQuery();
                    LogMessageToFile("SP_LoadJsonQueue executed.");
                    dbConnection.Close();
                }
                catch (Exception e)
                {
                    LogErrorMessage(e);
                    dbConnection.Close();
                }
            }
        }

        private void runPostMigrationScript()
        {
        //    string sqlConnectionString = System.Configuration.ConfigurationSettings.AppSettings["SQLServerConnectionString"];
            string sqlConnectionString = System.Configuration.ConfigurationManager.AppSettings["SQLServerConnectionString"];
            using (SqlConnection dbConnection = new SqlConnection(sqlConnectionString))
            {
                dbConnection.Open();
                SqlCommand sqlCmd = new SqlCommand("AHOBPR.SP_PostMigrationDataUpdates", dbConnection);
                try
                {
                    sqlCmd.CommandType = CommandType.StoredProcedure;
                    sqlCmd.ExecuteNonQuery();
                    LogMessageToFile("SP_PostMigrationDataUpdates executed.");
                }
                catch (Exception e)
                {
                    LogErrorMessage(e);
                }
            }
        }

        private void UpdateServiceMode(bool setToMigration)
        {
            String newMode = setToMigration ? "migration" : "normal";
            WebRequest request = WebRequest.Create(System.Configuration.ConfigurationManager.AppSettings["RestServiceUrl"].Replace("jsonparser/", "") + "setmigrationmode/" + newMode);
            request.Method = "GET";
            request.ContentType = @"application/json; charset=utf-8";
            String secretKey = System.Configuration.ConfigurationManager.AppSettings["secretKey"];
            request.Headers.Add("Authorization", "Basic " + secretKey);

            HttpWebResponse response = (HttpWebResponse)request.GetResponse();
            string jsonResponse = string.Empty;
            using (StreamReader sr = new StreamReader(response.GetResponseStream()))
            {
                jsonResponse = sr.ReadToEnd();
                LogMessageToFile("Response from REST service for set service mode to " + newMode + ": " + jsonResponse);
            }
        }
    }
}